#ifdef __linux__
/*
    Direct Hardware Access library
    Copyright (C) 1996  Robin Cutshaw (robin@xfree86.org)  XFree86 3.3.3 implementation
    Copyright (C) 1999	yvind Aabling. Modified for GATOS/win/gfxdump.
		  
    Copyright (C) 2002	Nick Kurshev. library implementation
    Copyright (C) 2002	Alex Beregszaszi. dhahelper and some changes
    Copyright (C) 2004  Dmitry Baryshkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "dha2.h"
#include "dha2_internal.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>

#include "kernelhelper/dhahelper.h"

static int dhahelper_fd = -1;

extern dha2_ports_driver_t dha2_dhahelper_ports;
extern dha2_mmap_driver_t dha2_dhahelper_mmap;

/* Two generic functions */
static int dha2_dhahelper_init(void)
{
	int api_version;
	errno = 0;

	if (dhahelper_fd != -1)
		return 0;

	dhahelper_fd = open ("/dev/dhahelper", O_RDWR);
	if (dhahelper_fd == -1)
	{
		dha2_dhahelper_ports.status = -1;
		dha2_dhahelper_mmap.status = -1;
		return -1;
	}

	ioctl (dhahelper_fd, DHAHELPER_GET_VERSION, &api_version);
	if (api_version != API_VERSION)
	{
		dha2_dhahelper_ports.status = -1;
		dha2_dhahelper_mmap.status = -1;

		close(dhahelper_fd);

		errno = EIO;
		return -1;
	}

	dha2_dhahelper_ports.status = 1;
	dha2_dhahelper_mmap.status = 1;
	return 0;
}

static void dha2_dhahelper_shutdown(void)
{
	if (dhahelper_fd != -1)
	{
		dha2_dhahelper_ports.status = 0;
		dha2_dhahelper_mmap.status = 0;

		close(dhahelper_fd);
		dhahelper_fd = -1;
	}
}

/* Port IO */

static unsigned char dha2_dhahelper_inb(unsigned port)
{
	dhahelper_port_t _port;
	_port.operation = PORT_OP_READ;
	_port.addr = port;
	_port.size = 1;
	if (ioctl(dhahelper_fd, DHAHELPER_PORT, &_port) == 0)
		return _port.value;
	else
		return 0xff;
}


static unsigned short dha2_dhahelper_inw(unsigned port)
{
	dhahelper_port_t _port;
	_port.operation = PORT_OP_READ;
	_port.addr = port;
	_port.size = 2;
	if (ioctl(dhahelper_fd, DHAHELPER_PORT, &_port) == 0)
		return _port.value;
	else
		return 0xffff;
}


static unsigned int dha2_dhahelper_inl(unsigned port)
{
	dhahelper_port_t _port;
	_port.operation = PORT_OP_READ;
	_port.addr = port;
	_port.size = 4;
	if (ioctl(dhahelper_fd, DHAHELPER_PORT, &_port) == 0)
		return _port.value;
	else
		return 0xffffffff;
}


static void dha2_dhahelper_outb(unsigned port, unsigned char value)
{
	dhahelper_port_t _port;
	_port.operation = PORT_OP_WRITE;
	_port.addr = port;
	_port.size = 1;
	_port.value = value;
	ioctl(dhahelper_fd, DHAHELPER_PORT, &_port);
}


static void dha2_dhahelper_outw(unsigned port, unsigned short value)
{
	dhahelper_port_t _port;
	_port.operation = PORT_OP_WRITE;
	_port.addr = port;
	_port.size = 2;
	_port.value = value;
	ioctl(dhahelper_fd, DHAHELPER_PORT, &_port);
}


static void dha2_dhahelper_outl(unsigned port, unsigned int value)
{
	dhahelper_port_t _port;
	_port.operation = PORT_OP_WRITE;
	_port.addr = port;
	_port.size = 4;
	_port.value = value;
	ioctl(dhahelper_fd, DHAHELPER_PORT, &_port);
}

dha2_ports_driver_t dha2_dhahelper_ports =
{
	.name = "DHA kernel helper (IO)",
	.status = 0,
	.init = dha2_dhahelper_init,
	.shutdown = dha2_dhahelper_shutdown,

	.dha2_inb = dha2_dhahelper_inb,
	.dha2_inw = dha2_dhahelper_inw,
	.dha2_inl = dha2_dhahelper_inl,

	.dha2_outb = dha2_dhahelper_outb,
	.dha2_outw = dha2_dhahelper_outw,
	.dha2_outl = dha2_dhahelper_outl,
};

/* Memory mapping/unmapping */

static void *dha2_dhahelper_map_phys_mem(unsigned long base, unsigned long size)
{
	dhahelper_memory_t mem_req;

	mem_req.operation = MEMORY_OP_MAP;
	mem_req.start = base;
	mem_req.offset = 0;
	mem_req.size = size;

	if (ioctl(dhahelper_fd, DHAHELPER_MEMORY, &mem_req) < 0)
		return MAP_FAILED;
					
	return mmap(0,size,PROT_READ|PROT_WRITE,MAP_SHARED,dhahelper_fd,base);
}

static void dha2_dhahelper_unmap_phys_mem(void *ptr, unsigned long size)
{
	munmap(ptr, size);
}

dha2_mmap_driver_t dha2_dhahelper_mmap =
{
	.name = "DHA kernel helper (mmap)",
	.status = 0,
	.init = dha2_dhahelper_init,
	.shutdown = dha2_dhahelper_shutdown,

	.map_phys_mem = dha2_dhahelper_map_phys_mem,
	.unmap_phys_mem = dha2_dhahelper_unmap_phys_mem,
};

#endif /* __linux__ */
