/*
    Direct Hardware Access library --- dha drivers wrapper.
    Copyright (C) 2004  Dmitry Baryshkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <errno.h>
#include <string.h>

#include "dha2.h"

#include "dha2_internal.h"

static int initialized = 0;

static dha2_ports_driver_t *dha2_ports = NULL;
static dha2_mtrr_driver_t *dha2_mtrr = NULL;
static dha2_pci_driver_t *dha2_pci = NULL;
static dha2_mmap_driver_t *dha2_mmap = NULL;

#ifdef __linux__
extern dha2_ports_driver_t dha2_svgahelper_ports;
extern dha2_pci_driver_t dha2_svgahelper_pci;
extern dha2_mmap_driver_t dha2_svgahelper_mmap;

extern dha2_ports_driver_t dha2_dhahelper_ports;
extern dha2_mmap_driver_t dha2_dhahelper_mmap;

extern dha2_pci_driver_t dha2_pci_proc_linux_pci;
#endif

#if defined(__i386__) && defined(__linux__)
extern dha2_mtrr_driver_t dha2_mtrr_linux_mtrr;
#endif
#if defined(__i386__) && defined(__NetBSD__)
extern dha2_mtrr_driver_t dha2_mtrr_netbsd_mtrr;
#endif

#if defined WIN32
extern dha2_mmap_driver_t dha2_win9x_mmap;
extern dha2_mmap_driver_t dha2_windha_mmap;
extern dha2_ports_driver_t dha2_windha_ports;
#elif defined __EMX__
extern dha2_mmap_driver_t dha2_os2_mmap;
#else
extern dha2_mmap_driver_t dha2_devmem_mmap;
#endif

#if defined(__alpha__)  || (defined(__powerpc__) && (defined(Lynx) || defined(__OpenBSD__)))
extern dha2_pci_driver_t dha2_pciconfig_pci;
#endif


extern dha2_pci_driver_t dha2_pci_generic_pci;
extern dha2_ports_driver_t dha2_ports_direct_ports;

/* Next arrays get filled with all available drivers */
static dha2_ports_driver_t *dha2_ports_drivers[] =
{
#ifdef __linux__
	&dha2_svgahelper_ports,
	&dha2_dhahelper_ports,
#endif
#ifdef WIN32
	&dha2_windha_ports,
#endif
	&dha2_ports_direct_ports,
	NULL
};

static dha2_mtrr_driver_t *dha2_mtrr_drivers[] =
{
#if defined(__i386__) && defined(__linux__)
	&dha2_mtrr_linux_mtrr,
#endif
#if defined(__i386__) && defined(__NetBSD__)
	&dha2_mtrr_netbsd_mtrr,
#endif
	NULL
};

static dha2_pci_driver_t *dha2_pci_drivers[] =
{
#ifdef __linux__
	&dha2_svgahelper_pci,
	&dha2_pci_proc_linux_pci,
#endif
#if defined(__alpha__)  || (defined(__powerpc__) && (defined(Lynx) || defined(__OpenBSD__)))
	&dha2_pciconfig_pci;
#endif
	&dha2_pci_generic_pci,
	NULL
};

static dha2_mmap_driver_t *dha2_mmap_drivers[] =
{
#ifdef __linux__
	&dha2_svgahelper_mmap,
	&dha2_dhahelper_mmap,
#endif
#if defined WIN32
	&dha2_win9x_mmap,
	&dha2_windha_mmap,
#elif defined (__EMX__)
	&dha2_os2_mmap,
#else
	&dha2_devmem_mmap,
#endif
	NULL
};

int init_drivers(void)
{
	int result = 0;
	int error = 0;
	int firsterror = 0;
	int i;
	
	if (initialized)
		return 0;

	for (i = 0; dha2_ports_drivers[i] != NULL; i++)
	{
		if (!dha2_ports_drivers[i]->init())
		{
			dha2_ports = dha2_ports_drivers[i];
			printf("Using '%s' as ports driver\n", dha2_ports_drivers[i]->name);
			break;
		}
		else
		{
			error = errno;
			fprintf(stderr, "'%s' init() returned '%s'\n", dha2_ports_drivers[i]->name, strerror(error));
			if (!firsterror)
				firsterror = error;
		}
	}

	if (!dha2_ports)
	{
		fprintf(stderr, "Can't find working ports driver\n");
		result |= DHA_PORTS_DRIVER;
	}

	for (i = 0; dha2_pci_drivers[i] != NULL; i++)
	{
		if (!dha2_pci_drivers[i]->init())
		{
			dha2_pci = dha2_pci_drivers[i];
			printf("Using '%s' as PCI driver\n", dha2_pci_drivers[i]->name);
			break;
		}
		else
		{
			error = errno;
			fprintf(stderr, "'%s' init() returned '%s'\n", dha2_pci_drivers[i]->name, strerror(error));
			if (!firsterror)
				firsterror = error;
		}
	}

	if (!dha2_pci)
	{
		fprintf(stderr, "Can't find working PCI driver\n");
		result |= DHA_PCI_DRIVER;
	}

	for (i = 0; dha2_mmap_drivers[i] != NULL; i++)
	{
		if (!dha2_mmap_drivers[i]->init())
		{
			dha2_mmap = dha2_mmap_drivers[i];
			printf("Using '%s' as mmap driver\n", dha2_mmap_drivers[i]->name);
			break;
		}
		else
		{
			error = errno;
			fprintf(stderr, "'%s' init() returned '%s'\n", dha2_mmap_drivers[i]->name, strerror(error));
			if (!firsterror)
				firsterror = error;
		}
	}

	if (!dha2_mmap)
	{
		fprintf(stderr, "Can't find working mmap driver\n");
		result |= DHA_MMAP_DRIVER;
	}

	for (i = 0; dha2_mtrr_drivers[i] != NULL; i++)
	{
		if (!dha2_mtrr_drivers[i]->init())
		{
			dha2_mtrr = dha2_mtrr_drivers[i];
			printf("Using '%s' as MTRR driver\n", dha2_mtrr_drivers[i]->name);
			break;
		}
		else
		{
			error = errno;
			fprintf(stderr, "'%s' init() returned '%s'\n", dha2_mtrr_drivers[i]->name, strerror(error));
			if (!firsterror)
				firsterror = error;
		}
	}

	if (!dha2_mtrr)
	{
		fprintf(stderr, "Can't find working MTRR driver\n");
		result |= DHA_MTRR_DRIVER;
	}

	errno = firsterror;

	initialized = 1;

	return result;
}

void close_drivers(void)
{
	initialized = 0;

	if (dha2_ports)
		dha2_ports->shutdown();

	if (dha2_mtrr)
		dha2_mtrr->shutdown();

	if (dha2_pci)
		dha2_pci->shutdown();

	if (dha2_mmap)
		dha2_mmap->shutdown();
}

unsigned char dha2_inb(unsigned port)
{
	if (!initialized)
		init_drivers();

	if (dha2_ports)
		return dha2_ports->dha2_inb(port);
	else
		return 0xff;
}

unsigned short dha2_inw(unsigned port)
{
	if (!initialized)
		init_drivers();

	if (dha2_ports)
		return dha2_ports->dha2_inw(port);
	else
		return 0xffff;
}

unsigned int dha2_inl(unsigned port)
{
	if (!initialized)
		init_drivers();

	if (dha2_ports)
		return dha2_ports->dha2_inl(port);
	else
		return 0xffffffff;
}

void dha2_outb(unsigned port, unsigned char value)
{
	if (!initialized)
		init_drivers();

	if (dha2_ports)
		dha2_ports->dha2_outb(port, value);
}

void dha2_outw(unsigned port, unsigned short value)
{
	if (!initialized)
		init_drivers();

	if (dha2_ports)
		dha2_ports->dha2_outw(port, value);
}

void dha2_outl(unsigned port, unsigned int value)
{
	if (!initialized)
		init_drivers();

	if (dha2_ports)
		dha2_ports->dha2_outl(port, value);
}

int mtrr_set_type(unsigned base, unsigned size, int type)
{
	if (!initialized)
		init_drivers();

	if (dha2_mtrr)
		return dha2_mtrr->mtrr_set_type(base, size, type);
	else
		return -ENOSYS;
}

int pci_config_type(void)
{
	if (!initialized)
		init_drivers();

	if (dha2_pci)
		return dha2_pci->pci_config_type();
	else
		return 0xFFFF;
}

int pci_get_vendor(unsigned char bus, unsigned char dev, int func)
{
	if (!initialized)
		init_drivers();

	if (dha2_pci)
		return dha2_pci->pci_get_vendor(bus, dev, func);
	else
		return 0xFFFF;
}

int pci_config_read_long(unsigned char bus, unsigned char dev, int func, unsigned cmd)
{
	if (!initialized)
		init_drivers();

	if (dha2_pci)
		return dha2_pci->pci_config_read_long(bus, dev, func, cmd);
	else
		return -ENOSYS;
}

void* map_phys_mem(unsigned long base, unsigned long size)
{
	if (!initialized)
		init_drivers();

	if (dha2_mmap)
		return dha2_mmap->map_phys_mem(base, size);
	else
		return MAP_FAILED;
}

void unmap_phys_mem(void * base, unsigned long size)
{
	if (!initialized)
		init_drivers();

	if (dha2_mmap)
		dha2_mmap->unmap_phys_mem(base, size);
}

int enable_app_io(void)
{
	/* mtrr isn't required! */	
	if (init_drivers() & (DHA_PORTS_DRIVER | DHA_PCI_DRIVER | DHA_MMAP_DRIVER))
		return errno;
	return 0;
}

int disable_app_io(void)
{
	if (initialized)
		close_drivers();
	return 0;
}

#ifdef WIN32
#include <windows.h>
int IsWinNT()
{
  OSVERSIONINFO OSVersionInfo;
  OSVersionInfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
  GetVersionEx(&OSVersionInfo);
  return OSVersionInfo.dwPlatformId == VER_PLATFORM_WIN32_NT;
}  
#else
int IsWinNT(){return 0;}
#endif
