#ifdef __linux__
/*
    Direct Hardware Access library
    Copyright (C) 1996  Robin Cutshaw (robin@xfree86.org)  XFree86 3.3.3 implementation
    Copyright (C) 1999	yvind Aabling. Modified for GATOS/win/gfxdump.
		  
    Copyright (C) 2002	Nick Kurshev. library implementation
    Copyright (C) 2002	Alex Beregszaszi. dhahelper and some changes
    Copyright (C) 2002  Matan Ziv-Av. Svgalib kernelhelper support
    Copyright (C) 2004  Dmitry Baryshkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "dha2.h"
#include "dha2_internal.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>

#include "svgalib_helper.h"

static int svgahelper_fd = -1;

extern dha2_ports_driver_t dha2_svgahelper_ports;
extern dha2_pci_driver_t dha2_svgahelper_pci;
extern dha2_mmap_driver_t dha2_svgahelper_mmap;

/* Two generic functions */
static int dha2_svgahelper_init(void)
{
	errno = 0;

	if (svgahelper_fd != -1)
		return 0;

	svgahelper_fd = open ("/dev/svga", O_RDWR);
	if (svgahelper_fd == -1)
	{
		dha2_svgahelper_ports.status = -1;
		dha2_svgahelper_pci.status = -1;
		dha2_svgahelper_mmap.status = -1;
		return -1;
	}

	dha2_svgahelper_ports.status = 1;
	dha2_svgahelper_pci.status = 1;
	dha2_svgahelper_mmap.status = 1;
	return 0;
}

static void dha2_svgahelper_shutdown(void)
{
	if (svgahelper_fd != -1)
	{
		dha2_svgahelper_ports.status = 0;
		dha2_svgahelper_pci.status = 0;
		dha2_svgahelper_mmap.status = 0;

		close(svgahelper_fd);
		svgahelper_fd = -1;
	}
}

/* Port IO */

static unsigned char dha2_svgahelper_inb(unsigned port)
{
	io_t iov;

	iov.port = port;
	if (ioctl(svgahelper_fd, SVGAHELPER_INB, &iov) == 0)
		return iov.val;
	else
		return 0xff;
}


static unsigned short dha2_svgahelper_inw(unsigned port)
{
	io_t iov;

	iov.port = port;
	if (ioctl(svgahelper_fd, SVGAHELPER_INW, &iov) == 0)
		return iov.val;
	else
		return 0xffff;
}


static unsigned int dha2_svgahelper_inl(unsigned port)
{
	io_t iov;

	iov.port = port;
	if (ioctl(svgahelper_fd, SVGAHELPER_INL, &iov))
		return iov.val;
	else
		return 0xffffffff;
}


static void dha2_svgahelper_outb(unsigned port, unsigned char value)
{
	io_t iov;

	iov.port = port;
	iov.val = value;
	ioctl(svgahelper_fd, SVGAHELPER_OUTB, &iov);
}


static void dha2_svgahelper_outw(unsigned port, unsigned short value)
{
	io_t iov;

	iov.port = port;
	iov.val = value;
	ioctl(svgahelper_fd, SVGAHELPER_OUTW, &iov);
}


static void dha2_svgahelper_outl(unsigned port, unsigned int value)
{
	io_t iov;

	iov.port = port;
	iov.val = value;
	ioctl(svgahelper_fd, SVGAHELPER_OUTL, &iov);
}

dha2_ports_driver_t dha2_svgahelper_ports =
{
	.name = "SVGAlib kernel helper (IO)",
	.status = 0,
	.init = dha2_svgahelper_init,
	.shutdown = dha2_svgahelper_shutdown,

	.dha2_inb = dha2_svgahelper_inb,
	.dha2_inw = dha2_svgahelper_inw,
	.dha2_inl = dha2_svgahelper_inl,

	.dha2_outb = dha2_svgahelper_outb,
	.dha2_outw = dha2_svgahelper_outw,
	.dha2_outl = dha2_svgahelper_outl,
};

/* PCI IO */
static int dha2_svgahelper_pci_config_type(void)
{
	return 1;
}

static int dha2_svgahelper_pci_config_read_long(unsigned char bus, unsigned char dev, int func, unsigned cmd)
{
	pcic_t p;

	p.address = cmd;
	p.pcipos = (bus << 8) | dev | (func << 5);
	
	if (!ioctl(svgahelper_fd, SVGAHELPER_PCIINL, &p))
		return p.val;
	else
		return 0xFFFFFFFF;
}

static int dha2_svgahelper_pci_get_vendor(unsigned char bus, unsigned char dev, int func)
{
	return dha2_svgahelper_pci_config_read_long(bus, dev, func, 0);
}

dha2_pci_driver_t dha2_svgahelper_pci =
{
	.name = "SVGAlib kernel helper (PCI)",
	.status = 0,
	.init = dha2_svgahelper_init,
	.shutdown = dha2_svgahelper_shutdown,

	.pci_config_type = dha2_svgahelper_pci_config_type,
	.pci_config_read_long = dha2_svgahelper_pci_config_read_long,
	.pci_get_vendor = dha2_svgahelper_pci_get_vendor,
};

/* Memory mapping/unmapping */

static void *dha2_svgahelper_map_phys_mem(unsigned long base, unsigned long size)
{
	return mmap(0,size,PROT_READ|PROT_WRITE,MAP_SHARED,svgahelper_fd,base);
}

static void dha2_svgahelper_unmap_phys_mem(void *ptr, unsigned long size)
{
	munmap(ptr, size);
}

dha2_mmap_driver_t dha2_svgahelper_mmap =
{
	.name = "SVGAlib kernel helper (mmap)",
	.status = 0,
	.init = dha2_svgahelper_init,
	.shutdown = dha2_svgahelper_shutdown,

	.map_phys_mem = dha2_svgahelper_map_phys_mem,
	.unmap_phys_mem = dha2_svgahelper_unmap_phys_mem,
};

#endif /* __linux__ */
