/*
 * mctun - mud compression tunnel.
 * Copyright (C) 2003 Dmitry Baryshkov <mitya@school.ioffe.ru>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <errno.h>

#ifndef WIN32
#include <signal.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#include <netdb.h>
#define closesocket close
#define WSAGetLastError() errno
#define WSAEINTR	EINTR
#define WSAEINPROGRESS	EINPROGRESS
#define WSAEWOULDBLOCK	EWOULDBLOCK
#define ioctlsocket	ioctl
#else
#include <winsock2.h>
#endif

#include <unistd.h>

#include "mccp.h"

#define MAXLISTEN	1
#define DEF_BUF_SIZE	1024
#define ZLIB_LEVEL	9

typedef struct redirect_s
{
	struct sockaddr_in *remote;

	int sd;

	int zlib_level;
	int wind_level;

	struct redirect_s *next;
} redirect;

typedef struct client_s
{
	int clientfd;
	int serverfd;

	char * to_server;
	char * to_client;

	int to_server_len;
	int to_client_len;

	int to_server_size;
	int to_client_size;

	mccp_handle *mc;

	int closing;

	struct client_s *next;
} client;

redirect *first_redir;
client *first_client;

void bailout(int s)
{
	fprintf(stderr, "Shutting down...\n");
#ifdef WIN32
	WSACleanup();
#endif
	exit(s);
}

void init()
{
#ifdef WIN32
	WSADATA wsadata;
#define WSVERS MAKEWORD(2,2)
#endif
	first_redir = NULL;
	first_client = NULL;

	fprintf(stderr, "Initializing...\n");

#ifdef WIN32
	if (WSAStartup(WSVERS, &wsadata) != 0)
	{
		fprintf(stderr, "Init of WSA failed :(\n");
		exit(1);
	}
#else
	signal(SIGTERM, bailout);
	signal(SIGQUIT, bailout);
	signal(SIGINT, bailout);
#endif
}

void readconfig ()
{
	FILE *fconf;
	char line[1024];
	char hostname[1024];
	short local_port, remote_port;
	short zliblev, memlev;

	if ((fconf = fopen ("mctun.cfg", "r")) == NULL)
	{
		fprintf(stderr, "Can't read config: mctun.cfg: %s\n", strerror(WSAGetLastError()));
	}
	while (fgets(line, 1024, fconf) != NULL)
	{
		if (line[0] == '#')
			continue;

		if (sscanf(line, "%hu%s%hu%hu%hu", &local_port, hostname, &remote_port, &zliblev, &memlev) != 5)
		{
			fprintf(stderr, "Error during parse of config file: line: '%s'\n", line);
			exit(2);
		}
		else
		{
			struct sockaddr_in *addr_s, *addr_l;
			int sock;
			int yes;
			redirect *redir;

			sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
			if (sock < 0)
			{
				fprintf(stderr, "Error creating socket: %s\n", strerror(WSAGetLastError()));
				exit(4);
			}

			yes = 1;
			if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &yes, sizeof (yes)) < 0)
			{
				perror ("setsockopt");
				closesocket (sock);
				exit(5);
			}
			addr_l = (struct sockaddr_in*)calloc(1, sizeof(struct sockaddr_in));
			addr_l->sin_family = PF_INET;
			addr_l->sin_port = htons( local_port );
			if (bind(sock, (struct sockaddr*)addr_l, sizeof(struct sockaddr_in)) < 0)
			{
				perror("bind");
				closesocket(sock);
				exit(6);
			}
			free(addr_l);

			listen(sock, MAXLISTEN);

			addr_s = (struct sockaddr_in*)calloc(1, sizeof(struct sockaddr_in));
			addr_s->sin_family = PF_INET;
			addr_s->sin_port = htons( remote_port );
#ifdef WIN32
			if ((addr_s->sin_addr.s_addr = inet_addr(hostname)) == INADDR_NONE)
#else
			if (inet_aton(hostname, &(addr_s->sin_addr)) == 0)
#endif
			{
				struct hostent *hp;

				fprintf(stderr, "Resolving %s...\n", hostname);
				if ((hp = gethostbyname(hostname)) == NULL)
				{
					fprintf(stderr, "Can't resolve: gethostbyname: %s\n", strerror(WSAGetLastError()));
					exit(7);
				}
				memcpy (&(addr_s->sin_addr), hp->h_addr, hp->h_length);
			}
			redir = (redirect*)calloc(1, sizeof(redirect));
			redir->remote = addr_s;
			redir->sd = sock;
			redir->zlib_level = zliblev;
			redir->wind_level = memlev;
			redir->next = first_redir;
			first_redir=redir;
			
		}
	}
	fclose(fconf);

	if (first_redir == NULL)
	{
		fprintf(stderr, "No redirects read. Terminating.\n");
		exit(8);
	}
}

void close_connection(client *cl)
{
	int stats_in, stats_out;
	if (cl->clientfd != -1)
	{
		closesocket(cl->clientfd);
	}

	if (cl->serverfd != -1)
	{
		closesocket(cl->serverfd);
	}
	free(cl->to_server);
	free(cl->to_client);

	if (cl->mc)
	{
		mccp_stats(cl->mc, &stats_in, &stats_out);
		fprintf(stderr, "     Received %d bytes, send %d bytes. Saved %5.1f%% of bandwidth.\n", stats_in, stats_out, (stats_in==0)?0:((stats_in-stats_out)*100.0/stats_in));

		mccp_finalize(cl->mc);
	}

	if (first_client == cl)
		first_client = cl->next;
	else
	{
		client * cl_temp;
		for (cl_temp = first_client; cl_temp != NULL; cl_temp = cl_temp->next)
			if (cl_temp->next == cl)
				cl_temp->next = cl->next;
	}
	free(cl);
}

void accept_new(redirect *cur_red)
{
	int sock, remote;
	int size;
	struct sockaddr_in *addr;
	client * cl;
	addr = calloc(1, sizeof(struct sockaddr_in));
	size = sizeof(struct sockaddr_in);
	sock = accept(cur_red->sd, (struct sockaddr*)addr, &size);
	unsigned long yes = 1;

	if (sock < 0)
	{
		perror("accept");
		return;
	}
	fprintf(stderr, "Accepted new connection from %s:%d.\n", inet_ntoa(addr->sin_addr), addr->sin_port);
	ioctlsocket(sock, FIONBIO, &yes);

	remote = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (remote == -1)
	{
		fprintf(stderr, "socket: %s\n", strerror(WSAGetLastError()));
		closesocket(sock);
		return;
	}
	if (connect(remote, (struct sockaddr*)cur_red->remote, sizeof(struct sockaddr_in)) && WSAGetLastError() != WSAEINPROGRESS)
	{
		fprintf(stderr, "connect(remote): %s\n", strerror(WSAGetLastError()));
		closesocket(remote);
		closesocket(sock);
		return;
	}
	ioctlsocket(remote, FIONBIO, &yes);
	cl = calloc(1, sizeof(client));
	cl->clientfd = sock;
	cl->serverfd = remote;
	cl->to_server_len = cl->to_client_len = 0;
	cl->to_server_size = cl->to_client_size  = DEF_BUF_SIZE;
	cl->to_server = calloc(cl->to_server_size, sizeof(char));
	cl->to_client = calloc(cl->to_client_size, sizeof(char));
	cl->closing = 0;

	if ((cl->mc = mccp_init(cur_red->zlib_level, cur_red->wind_level)) == NULL)
	{
		fprintf(stderr, "Cannot initialize mccp library.");
		close_connection(cl);
		return;
	}

	cl->next = first_client;
	first_client = cl;
	free(addr);
}

#define grow_buf(bufbase, str, str_size)				\
{									\
	if (bufbase ## _len + str_size > bufbase ## _size) 		\
	{								\
		bufbase = realloc (bufbase, 				\
			bufbase ## _size = bufbase ## _len + str_size);	\
	}								\
	memcpy(bufbase + bufbase ## _len, str, str_size);		\
	bufbase ## _len += str_size;					\
}

int exception_client(client *cl){fprintf(stderr, "FIXME: Function %s is stub!!!\n", __FUNCTION__);return 1;}
int exception_server(client *cl){fprintf(stderr, "FIXME: Function %s is stub!!!\n", __FUNCTION__);return 1;}

int read_from_client(client *cl)
{
	char buf[DEF_BUF_SIZE];
	int n;
	char *after_mccp;
	int size;

	for (;;)
	{
		n = recv(cl->clientfd, buf, sizeof(buf), 0);
		if (n == 0)
		{
			/* EOF encountered */
			close_connection(cl);
			return 1;
		}
		else if (n < 0)
		{
			if (WSAGetLastError() != WSAEWOULDBLOCK
#ifndef WIN32
					&& errno != EAGAIN
#endif
			   )
			{
				perror("recv");
				close_connection(cl);
				return 1;
			}
			return 0;
		}
		size = mccp_recv(cl->mc, buf, n, &after_mccp);
		grow_buf(cl->to_server, after_mccp, size);
		free(after_mccp);
	}
}

int read_from_server(client *cl)
{
	char buf[DEF_BUF_SIZE];
	int n;
	char *after_mccp;
	int size;

	for (;;)
	{
		n = recv(cl->serverfd, buf, sizeof(buf), 0);
		if (n == 0)
		{
			/* EOF encountered */
			cl->closing = 1;
			return 0;
		}
		else if (n < 0)
		{
			if (WSAGetLastError() != WSAEWOULDBLOCK
#ifndef WIN32
					&& errno != EAGAIN
#endif
			   )
			{
				perror("recv");
				close_connection(cl);
				return 1;
			}
			return 0;
		}
		size = mccp_send(cl->mc, buf, n, &after_mccp);
		grow_buf(cl->to_client, after_mccp, size);
		free(after_mccp);
		
	}
}

int write_to_client(client *cl){
	int n, sent = 0;
	for (;;)
	{
		n = send(cl->clientfd, cl->to_client + sent, cl->to_client_len - sent,0);
		if (n == 0 || (n < 0 && (WSAGetLastError() == WSAEWOULDBLOCK
#ifndef WIN32
				|| errno == EAGAIN
#endif
				)))
		{
			if (cl->to_client_len == sent)
			{
				cl->to_client_len = 0;
				if (cl->closing)
				{
					close_connection(cl);
					return 1;
				}
				return 0;
			}
			memmove(cl->to_client, cl->to_client + sent, cl->to_client_len - sent);
			return 0;
		}
		else if (n < 0)
		{
			close_connection(cl);
			return 1;
		}
		else
			sent += n;
	}
}

int write_to_server(client *cl)
{
	int n, sent = 0;
	for (;;)
	{
		n = send(cl->serverfd, cl->to_server + sent, cl->to_server_len - sent,0);
		if (n == 0 || (n < 0 && (WSAGetLastError() == WSAEWOULDBLOCK
#ifndef WIN32
				|| errno == EAGAIN
#endif
				)))
		{
			if (cl->to_server_len == sent)
			{
				cl->to_server_len = 0;
				return 0;
			}
			memmove(cl->to_server, cl->to_server + sent, cl->to_server_len - sent);
			return 0;
		}
		else if (n < 0)
		{
			close_connection(cl);
			return 1;
		}
		else
			sent += n;
	}
}

void loop()
{
	while (1)
	{
		fd_set rd, wr, er;
		redirect *cur_red;
		client *cur_cli, *cur_cli_next;
		int r, n = 0;

		FD_ZERO(&rd);
		FD_ZERO(&wr);
		FD_ZERO(&er);
		for (cur_red = first_redir; cur_red != NULL; cur_red = cur_red->next)
		{
			if (cur_red->sd > n)
				n = cur_red->sd;
			FD_SET(cur_red->sd, &rd);
		}
		for (cur_cli = first_client; cur_cli != NULL; cur_cli = cur_cli->next)
		{
			if (cur_cli->clientfd > n)
				n = cur_cli->clientfd;
			if (cur_cli->serverfd > n)
				n = cur_cli->serverfd;

			FD_SET(cur_cli->clientfd, &rd);
			FD_SET(cur_cli->serverfd, &rd);

			if (cur_cli->to_client_len > 0)
				FD_SET(cur_cli->clientfd, &wr);
			if (cur_cli->to_server_len > 0)
				FD_SET(cur_cli->serverfd, &wr);

			FD_SET(cur_cli->clientfd, &er);
			FD_SET(cur_cli->serverfd, &er);
		}
		r = select(n+1, &rd, &wr, &er, NULL);
		if (r == -1)
		{
			if (WSAGetLastError() != WSAEINTR)
				continue;
			else
				bailout(9);
		}
		for (cur_red = first_redir; cur_red != NULL; cur_red = cur_red->next)
		{
			if (FD_ISSET(cur_red->sd, &rd))
			{
				accept_new(cur_red);
			}
		}
		for (cur_cli = first_client; cur_cli != NULL; cur_cli = cur_cli_next)
		{
			cur_cli_next = cur_cli->next;
			if (FD_ISSET(cur_cli->clientfd, &er))
				if (exception_client(cur_cli))
					continue;
			if (FD_ISSET(cur_cli->serverfd, &er))
				if (exception_server(cur_cli))
					continue;
			if (FD_ISSET(cur_cli->clientfd, &rd))
				if (read_from_client(cur_cli))
					continue;
			if (FD_ISSET(cur_cli->serverfd, &rd))
				if (read_from_server(cur_cli))
					continue;
			if (cur_cli->closing || FD_ISSET(cur_cli->clientfd, &wr))
				if (write_to_client(cur_cli))
					continue;
			if (FD_ISSET(cur_cli->serverfd, &wr))
				if (write_to_server(cur_cli))
					continue;
		}
	}

	
}

int main ()
{
	init();
	readconfig();
	fprintf(stderr, "Starting redirection...\n");
	loop();
	bailout(0);
	return 0;
}
