/*
 * mctun - mud compression tunnel.
 * Copyright (C) 2003 Dmitry Baryshkov <mitya@school.ioffe.ru>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <zlib.h>

#include "mccp.h"

struct mccp_handle_s
{
	z_stream *zstr;
	int level;
	int wbits;
	int state;
	char *to_send;
	int to_send_size;
	int pending;
	int total_in;
	int total_out;
};

#define	IAC			255
#define	DONT			254
#define	DO			253
#define	WONT			252
#define	WILL			251
#define	SB			250
#define	SE			240

#define TELOPT_COMPRESS1	85
#define TELOPT_COMPRESS2	86


mccp_handle * mccp_init(int level, int windowBits)
{
	mccp_handle *mc;
	char wills[] = {IAC, WILL, TELOPT_COMPRESS2, IAC, WILL, TELOPT_COMPRESS1};
	if (level == -2)
	{
//#warning FIXME: decompression not supported (yet)
		fprintf(stderr, "Decompression not supported (yet).\n");
		return NULL;
	}
	else if (level > 9 || level < -2)
	{
		fprintf(stderr, "bad level value: %d\n", level);
		return NULL;
	}
	if (windowBits > MAX_WBITS)
	{
		fprintf(stderr, "windowBits too high (can't not be greater, than %d).\n", MAX_WBITS);
		return NULL;
	}
	mc = calloc(1, sizeof(mccp_handle));
	mc->zstr = NULL;
	mc->level = level;
	mc->wbits = windowBits;
	mc->state = 0;
	mc->to_send = malloc(6);
	mc->to_send_size = 6;
	mc->total_in = mc->total_out = 0;
	memcpy(mc->to_send, wills, 6);
	return mc;
}

void mccp_finalize(mccp_handle *mc)
{
	if (mc)
	{
		if (mc->zstr)
		{
			free(mc->zstr);
			deflateEnd(mc->zstr);
		}
		if (mc->to_send)
			free(mc->to_send);
		free(mc);
	}
}


int mccp_recv(mccp_handle *mc, char *data, int len, char**answer)
{
	char *cur_ans;
	const char *cur_data;
	int dev;
	int flag = 0;
	/* Quite a hack, but processed data size is not greater, than incoming + 2 (pending from previous) */
	cur_ans = *answer = malloc(len+2);
	dev = 0;

	if (!mc)
	{
		return -1;
	}

	if (mc->pending != 0)
	{
		char *d_temp;
		flag = 1;
		d_temp = malloc(len+mc->pending);
		memcpy(d_temp+mc->pending, data, len);
		*d_temp = IAC;
		if (mc->pending == 2)
			*(d_temp+1) = DO;
		else if (mc->pending == -2)
			*(d_temp+1) = DONT;
		data = d_temp;
		len = len+mc->pending;
		mc->pending = 0;
	}
	cur_data = data;
	for (; cur_data + dev < data +len;)
	{
		if (*(cur_data+dev) != (char)IAC)
		{
			*cur_ans = *(cur_data+dev);
			cur_ans ++;
			cur_data ++;
		}
		else
		{
			int avail = data + len - (cur_data+dev)-1;
			if (avail >= 1)
			{
				if (*(cur_data+dev+1) == (char)IAC)
				{
					*cur_ans = *(cur_ans+1) = IAC;
					cur_ans +=2;
					cur_data +=2;
				}
				else if (*(cur_data+dev+1) == (char)DO)
				{
					if (avail >= 2)
					{
						if (*(cur_data+dev+2) == (char)TELOPT_COMPRESS2 ||*(cur_data+dev+2) == (char)TELOPT_COMPRESS1)
						{
							int ret;
							mc->zstr = calloc(1, sizeof(z_stream));
							mc->zstr->zalloc = Z_NULL; /*Use zlib internals*/
							mc->zstr->zfree = Z_NULL;
							mc->zstr->opaque = NULL;
							mc->zstr->next_in = NULL;
							mc->zstr->next_out = NULL;
							if ((ret = deflateInit2(mc->zstr, mc->level, Z_DEFLATED, mc->wbits, 8 , Z_DEFAULT_STRATEGY)) != Z_OK)
							{
								fprintf(stderr, "deflateInit2 failed.\n");
								free(mc->zstr);
								mc->zstr = NULL;
								
							}
							else
							{
								mc->to_send = (mc->to_send_size!=0)?realloc(mc->to_send, mc->to_send_size+5):malloc(5);
								*(mc->to_send+mc->to_send_size+0) = IAC;
								*(mc->to_send+mc->to_send_size+1) = SB;
								*(mc->to_send+mc->to_send_size+2) = *(cur_data+dev+2);
								*(mc->to_send+mc->to_send_size+3) = (*(cur_data+dev+2) == (char)TELOPT_COMPRESS2)?IAC:WILL;
								*(mc->to_send+mc->to_send_size+4) = SE;
								mc->to_send_size += 5;
								fprintf(stderr, "Compression negotiated.\n");
							}
							dev += 3;
						}
						else
						{
							memcpy(cur_ans, cur_data+dev, 3);
							cur_ans += 3;
							cur_data += 3;
						}
					}
					else
						mc->pending=2;
				}
				else if (*(cur_data+dev+1) == (char)DONT)
				{
					if (avail >= 2)
					{
						if (*(cur_data+dev+2) == (char)TELOPT_COMPRESS2)
						{
							dev += 3;
						}
						else if (*(cur_data+dev+2) == (char)TELOPT_COMPRESS1)
						{
							dev += 3;
						}
						else
						{
							memcpy(cur_ans, cur_data+dev, 3);
							cur_ans += 3;
							cur_data += 3;
						}
					}
					else
						mc->pending=-2;
				}
				else
				{
					memcpy(cur_ans, cur_data+dev, 2);
					cur_ans += 2;
					cur_data += 2;
				}
			}
			else
			{
				mc->pending = 1;
			}
		}
	}
	if (flag)
		free(data);
	return cur_ans - *answer;
}

int mccp_send(mccp_handle *mc, char *data, int len, char**answer)
{
	char * ans = NULL;
	int ans_size = 0;
	char *temp;
	int temp_size;
	int flag = 0;

	mc->total_in += len;
	if (!mc)
	{
		return -1;
	}

	if (mc->zstr == NULL)
	{
		temp = data;
		temp_size = len;
	}
	else
	{
#define BSIZE	1024
		mc->zstr->next_in = data;
		mc->zstr->avail_in = len;
		temp = malloc(BSIZE);
		flag = 1;
		mc->zstr->next_out = temp;
		mc->zstr->avail_out = BSIZE;
		temp_size = BSIZE;
		while (mc->zstr->avail_in)
		{
			int ret;
			if ((ret = deflate(mc->zstr, 0)) != Z_OK)
			{
				fprintf(stderr, "Deflate returned error: %s", zError(ret));
				break;
			}
			if (mc->zstr->avail_out == 0)
			{
				temp = realloc(temp, temp_size + BSIZE);
				mc->zstr->next_out = temp+temp_size;
				mc->zstr->avail_out = BSIZE;
				temp_size += BSIZE;
			}
		}

		deflate(mc->zstr, Z_SYNC_FLUSH);
		while (mc->zstr->avail_out == 0)
		{
				temp = realloc(temp, temp_size + BSIZE);
				mc->zstr->next_out = temp+temp_size;
				mc->zstr->avail_out = BSIZE;
				temp_size += BSIZE;
				deflate(mc->zstr, Z_SYNC_FLUSH);
		}
		temp_size -= mc->zstr->avail_out;
	}
	ans = malloc(temp_size + mc->to_send_size);
	if (mc->to_send_size)
		memcpy(ans, mc->to_send, mc->to_send_size);
	memcpy(ans+mc->to_send_size, temp, temp_size);
	ans_size = mc->to_send_size + temp_size;
	mc->to_send_size = 0;
	free(mc->to_send);
	mc->to_send = NULL;
	if (flag)
		free(temp);
	mc->total_out += ans_size;
	*answer = ans;
	return ans_size;
}

void mccp_stats(mccp_handle *mc, int *ret_in, int *ret_out)
{
	if (!mc)
	{
		*ret_in = *ret_out = 0;
	}
	else
	{
		*ret_in = mc->total_in;
		*ret_out = mc->total_out;
	}
}
